/* global sc */

// Load generic string prototypes
require('/agorum/roi/customers/cdev.ncore/js/utils/string');

let DEBUG = false; // debug the debug module itself

function createModule(config) {
  // Use a closure to cache the module instance
  let moduleInstance = {};

  let environment = config || {}; // Environment object

  // Console may not be available, e.g., in custom service requests
  let _print, stringify, checkDebug, enable, disable, list, clear, cache;

  function _checkDebugEntry(debug, environment) {
    DEBUG && console.log('checkDebugEntry', debug, environment);
    for (let i = 0; i < debug.length; i++) {
      let entry = debug[i];
      if (entry.path && entry.path === environment.path) {
        return true;
      } else if (
        entry.package &&
        entry.file &&
        entry.package === environment.package &&
        entry.file === environment.file
      ) {
        return true;
      } else if (entry.package && entry.package === environment.package) {
        return true;
      } else if (entry.name && entry.name === environment.name) {
        return true;
      }
    }
    return false;
  }

  function _addDebugEntry(debug, entry) {
    DEBUG && console.log('addDebugEntry', debug, entry);
    if (!_checkDebugEntry(debug, entry)) {
      debug.push(entry);
    }
    return debug;
  }

  function _deleteDebugEntry(debug, entry) {
    DEBUG && console.log('deleteDebugEntry', debug, entry);
    for (let i = 0; i < debug.length; i++) {
      if (_checkDebugEntry([debug[i]], entry)) {
        debug.splice(i, 1);
        break;
      }
    }
    return debug;
  }

  function _getPrefix(environment) {
    let prefix = null;
    if (environment.prefix) {
      if (environment.prefix.indexOf('{') !== -1) {
        prefix = environment.prefix.format(environment);
      } else {
        prefix = environment.prefix;
      }
    }
    return prefix;
  }

  // Detect environment
  let isNode = typeof process !== 'undefined' && process.versions && process.versions.node;

  if (isNode) {
    ////////////////////////////////////////////
    // Setup Node.js based runtime environment //
    ////////////////////////////////////////////
    environment.name = 'node';
    enable = (entry) => {
      process.env[entry.name] = 'true';
    };
    disable = (entry) => {
      process.env[entry.name] = 'false';
    };
    // Environment-specific check for debug flag
    checkDebug = (environment) => {
      if (!environment.debug) return false;
      if (typeof environment.debug === 'string') {
        return process.env[environment.debug] === 'true';
      }
      return !!environment.debug;
    };
    stringify = (obj) => {
      return JSON.stringify(obj, null, 2);
    };
    _print = console.log;
  } else {
    //////////////////////////////////////////////////
    // Setup agorum/Rhino based runtime environment //
    //////////////////////////////////////////////////
    let objects = require('common/objects').sc(environment.context || sc);
    let property = require('common/property');
    cache = require('common/cache')('CDEV_NCORE');
    environment.name = 'agorum';
    console.log('environment.path', environment.path);
    if (environment.path) {
      if (environment.path === 'console') {
        // use path: 'console' to refer to cdev/console widget
        environment.path = '/agorum/roi/customers/cdev.console/aguila/aguila-console.js';
      } else {
        // check for path literally or via module.id lookup
        let pathObj = objects.tryFind(environment.path);
        environment.path = pathObj ? pathObj.anyFolderPath : null;
      }
    }

    enable = (entry) => {
      DEBUG && console.log('enable', entry);
      cache.write('debug', _addDebugEntry(cache.read('debug') || [], entry));
    };

    disable = (entry) => {
      DEBUG && console.log('disable', entry);
      cache.write('debug', _deleteDebugEntry(cache.read('debug') || [], entry));
    };

    list = () => {
      DEBUG && console.log('list', cache.read('debug'));
      return cache.read('debug') || [];
    };

    clear = () => {
      cache.write('debug', []);
    };

    // Environment-specific check for debug flag
    checkDebug = (environment) => {
      if (!environment.debug) return false;
      if (typeof environment.debug === 'string') {
        if (environment.debug === 'cache') {
          // Check debug entry from agorum specific cache settings
          return _checkDebugEntry(cache.read('debug') || [], environment);
        }
        // Legacy: check for cache.read('PACKAGE') == true
        return !!cache.read(environment.debug);
      }
      // Evaluate boolean debug flag
      return !!environment.debug;
    };

    // Use console log for diagnostic output, check if log-channel is available
    try {
      _print = require('/agorum/roi/customers/cdev.console/js/log-channel').log;
    } catch (err) {
      try {
        _print = require('/agorum/roi/customers/cdev.core/js/log-channel').log;
      } catch (err) {
        _print = console && console.log ? console.log : () => null;
      }
    }

    stringify = (obj) => {
      if (obj === null) return 'null';
      if (typeof obj === 'object') {
        if (obj.className && obj.name && obj.ID) {
          return property
            .from({
              name: obj.name,
              ID: obj.ID,
              UUID: obj.UUID,
              className: obj.className,
            })
            .toPrettyString();
        } else {
          try {
            return property.from(obj).toPrettyString();
          } catch (err) {
            return 'null';
          }
        }
      }
      return obj;
    };
  }

  //////////////////////////////////////////
  // End of environment specific settings //
  //////////////////////////////////////////

  // Split a path like /agorum/roi/customers/cdev.ncore/js/aguila/ncore.js
  if (environment.path) {
    let tokens = environment.path.trim().split('/');
    environment.package = tokens[4];
    environment.file = tokens[tokens.length - 1].replace('.js', '');
  }

  DEBUG && console.log('agorum environment', environment);

  /**
   * Console log wrapper (without debug flag checking)
   * @param {...*} var_args - variable list of arguments
   */
  function print() {
    DEBUG && console.log('print', arguments);
    let prefix = _getPrefix(environment);
    if (!isNode) {
      // Stringify agorum objects
      let out = prefix ? [prefix] : [];
      for (let i = 0; i < arguments.length; i++) {
        let arg = arguments[i];
        out.push(typeof arg === 'object' ? stringify(arg) : arg);
      }
      _print(out.join(' '));
    } else {
      // Just forward arguments to console.log
      let outArgs = prefix ? [prefix] : [];
      for (let i = 0; i < arguments.length; i++) {
        outArgs.push(arguments[i]);
      }
      _print.apply(null, outArgs);
    }
  }

  /**
   * Console log wrapper (checks debug flag)
   * @param {...*} var_args - variable list of arguments
   */
  function debug() {
    if (checkDebug(environment)) {
      DEBUG && console.log('debug', arguments);
      print.apply(null, arguments);
    }
  }

  /**
   * Logs messages to the appropriate log channel
   * @param {string} msg - The message to log
   */
  function log(msg) {
    let logMsg = environment.package + '[' + global.sc.loginUser.name + ']: ' + msg;
    if (environment.name === 'agorum' && environment.package) {
      DEBUG && console.log('log', logMsg);
      global.Packages.agorum.commons.logging.Log.info(logMsg);
    } else {
      print(logMsg);
    }
  }

  // Assign properties to the module instance
  Object.assign(moduleInstance, {
    environment: environment,
    print: print,
    debug: debug,
    log: log,
    enable: enable,
    disable: disable,
    list: list,
    clear: clear,
    cache: cache,
  });

  return moduleInstance;
}

// Create and cache the default module instance
let defaultModule = createModule();

// Export the module as both a function and an object (legacy mode)
let moduleExport = (config) => {
  // If no config is provided, return the default module
  if (!config) return defaultModule;
  // Otherwise, create a new module instance with the provided config
  return createModule(config);
};

// Merge the default module properties into the exported function
Object.assign(moduleExport, defaultModule);

module.exports = moduleExport;
